from datetime import datetime as dt
from typing import List

import pandas as pd
from py_jftech import component, autowired, get_config, format_date, filter_weekend

from api import RoboReportor


@component(bean_name='fixed-range-report')
class FixedRangeReport(RoboReportor):

    @autowired(names={'combo': 'combo-report'})
    def __init__(self, combo: RoboReportor = None):
        self._combo = combo
        self._config = get_config(__name__)

    @property
    def report_name(self) -> str:
        return '固定区间收益率'

    @property
    def range_dates(self):
        return self._config['range-dates']

    def load_report(self, max_date=dt.today(), min_date=None) -> List[dict]:
        datas = pd.DataFrame(self._combo.load_report(max_date=max_date, min_date=min_date))
        if not datas.empty:
            datas.set_index('date', inplace=True)
            result = pd.DataFrame(columns=datas.columns)
            for range in self.range_dates:
                start = filter_weekend(range['start'])
                end = filter_weekend(range['end'])
                if not datas[start:end].empty:
                    row_name = f"{format_date(start)}~{format_date(end)}"
                    result.loc[row_name] = datas[start:end].values[-1] / datas[start:end].values[0] - 1

            result = round(result, 4) * 100
            result.reset_index(inplace=True)
            result.rename(columns={'index': 'range-date'}, inplace=True)
            return result.to_dict('records')
        return []
